/*    imaginary.sql.iMsqlResultSet
 *    from the Imaginary Java Class Library
 *    Copyright (c) 1996 George Reese
 *    created by George Reese (borg@imaginary.com) 960311
 *    mSQL implementation of the JDBC draft protocol ResultSet interface
 */

package imaginary.sql;

import java.sql.Date;
import java.sql.Numeric;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.SQLWarning;
import java.sql.Time;
import java.sql.Timestamp;
import java.sql.Types;
import java.util.Hashtable;
import imaginary.sql.iMsqlMetaResult;
import imaginary.sql.msql.MsqlException;
import imaginary.sql.msql.MsqlFieldDesc;
import imaginary.sql.msql.MsqlResult;

/**
 * This class implements the JDBC draft specification for the ResulSet
 * interface for mSQL.  The class is designed to hold the results of
 * database queries.
 *
 * @see java.sql.ResultSet
 * @version 1.9, 96/06/06
 * @author George Reese
 */
public class iMsqlResultSet implements java.sql.ResultSet {
  /**
   * Darryl Collins' MsqlResult object for this query
   */
  private MsqlResult result;
  /**
   * The current row data
   */
  private String current_row[];
  /**
   * Field information for the current row
   */
  MsqlFieldDesc current_fields[];
  /* Row number tracking removed in Imaginary 0.92 */
  /**
   * The meta data for this result set.
   */
  private iMsqlMetaResult meta;
  /**
   * A Hashtable that maps column names to columns
   */
  private Hashtable column_map = null;
  /**
   * Keeps track of the last column read.  This is efficient for mSQL since
   * no in place updates are allowed in mSQL.
   */
  private int last_read = -1;
  
  /**
   * Constructs a new result set object given the MsqlResult specified.
   * @param res the MsqlResult returned from a previously executed query
   */
  public iMsqlResultSet(MsqlResult res) {
    result = res;
  }

  /**
   * JDBC draft specification method for moving the current row to the
   * next row, returning true if there is another row for processing.
   * @see java.sql.ResultSet#next
   * @exception SQLException thrown if an error occurs during processing
   * @return true if there are more rows to process, otherwise false
   */
  public synchronized boolean next() throws SQLException {
    try {
      last_read = -1;
      if( result.NumRows() < 1 ) {
	current_row = null;
	current_fields = null;
	return false;
      }
      if( (current_row = result.FetchRow()) == null ) {
	current_fields = null;
	return false;
      }
      current_fields = result.ListFields();
      return true;
    }
    catch( Exception e ) {
      throw new SQLException(e.getMessage());
    }
  }
    
  /**
   * JDBC specification method to determine if a column is null.
   * @see java.sql.ResultSet#isNull
   * @exception SQLException in the event of an MsqlException
   * @param column the column being tested for nullness
   * @return true if the column is null, false otherwise
   */
  public boolean isNull(int column) throws SQLException {
    try {
      return (current_fields[column-1].FieldType() == 5);
    }
    catch( Exception e ) {
      throw new SQLException(e.getMessage());
    }
  }

  /**
   * JDBC specification method to determine if the last column read was null.
   * @see java.sql.ResultSet#wasNull
   * @exception SQLException in the event of an MsqlException
   * @return true if the column was null, false otherwise
   */
  public boolean wasNull() throws SQLException {
    if( last_read == -1 ) return false;
    return isNull(last_read);
  }

  /**
   * JDBC draft specification method for getting a string value from
   * the named column.  Note that the JDBC draft provides that this
   * method gets the value as a char, so you can retrieve int values
   * into String objects.
   * @see java.sql.ResultSet#getString
   * @exception SQLException thrown for invalid columns or bad rows
   * @param column the column being retrieved
   * @return the column as a String
   */
  public String getString(int column) throws SQLException {
    last_read = column;
    if( isNull(column) ) return null;
    try {
      return new String(current_row[column - 1]);
    }
    catch( NullPointerException e ) {
      return null;
    }
    catch( Exception e ) {
      throw new SQLException(e.getMessage());
    }
  }

  /**
   * Get the value of a column in the current row as a Java byte.
   *
   * @param columnIndex the first column is 1, the second is 2, ...
   * @return the column value; if the value is SQL NULL the result is 0
   */
  public byte getByte(int columnIndex) throws SQLException {
    String str;
    
    last_read = columnIndex;
    if( isNull(columnIndex) ) return 0;
    str = getString(columnIndex);
    if( str.equals("") ) return 0;
    else return (byte)str.charAt(0);
  }
  
  /**
   * JDBC specification method for retrieving a column as a boolean
   * value.  Interprets "", null, and "0" as false, others as true.
   * @see java.sql.ResultSet#getBoolean
   * @exception SQLException a sure sign of the apocolypse
   * @param column the column for which the value is being retrieved
   * @return false for "", null, or "0"; true otherwise
   */
  public boolean getBoolean(int column) throws SQLException {
    last_read = column;
    if( isNull(column) ) return false;
    try {
      String str = current_row[column-1];
      
      if( str.equals("") ) return false;
      if( str.equals("0") ) return false;
      return true;
    }
    catch( Exception e ) {
      throw new SQLException(e.getMessage());
    }
  }

  /**
   * JDBC draft specification method to retrieve a short value from
   * the database.
   * @see java.sql.ResultSet#getShort
   * @exception SQLException things did not go so hot
   * @param column the column being retrieved
   * @return the named column as a short
   */
  public short getShort(int column) throws SQLException {
    String str;

    str = getString(column);
    if( str == null ) return 0;
    try {
      return (short)(Integer.valueOf(str).intValue());
    }
    catch( NumberFormatException e ) {
      throw new SQLException(e.getMessage());
    }
  }  

  /**
   * JDBC draft specification method to retrieve an integer value from
   * the database.
   * @see java.sql.ResultSet#getInt
   * @exception SQLException things did not go so hot
   * @param column the column being retrieved
   * @return the named column as an integer
   */
  public int getInt(int column) throws SQLException {
    String str;

    str = getString(column);
    if( str == null ) return 0;
    try {
      return Integer.valueOf(str).intValue();
    }
    catch( NumberFormatException e ) {
      throw new SQLException(e.getMessage());
    }
  }  
  
  /**
   * JDBC draft specification method to retrieve a long value from
   * the database.
   * @see java.sql.ResultSet#getLong
   * @exception SQLException things did not go so hot
   * @param column the column being retrieved
   * @return the named column as a short
   */
  public long getLong(int column) throws SQLException {
    String str;

    str = getString(column);
    if( str == null ) return 0;
    try {
      return Long.valueOf(str).longValue();
    }
    catch( NumberFormatException e ) {
      throw new SQLException(e.getMessage());
    }
  }  

  /**
   * JDBC draft specification method to retrieve a float value from
   * the database.
   * @see java.sql.ResultSet#getFloat
   * @exception SQLException things did not go so hot
   * @param column the column being retrieved
   * @return the named column as a float
   */
  public float getFloat(int column) throws SQLException {
    String str;

    str = getString(column);
    if( str == null ) return 0;
    try {
      return Float.valueOf(str).floatValue();
    }
    catch( NumberFormatException e ) {
      throw new SQLException(e.getMessage());
    }
  }  

  /**
   * JDBC draft specification method to retrieve a double value from
   * the database.
   * @see java.sql.ResultSet#getDouble
   * @exception SQLException things did not go so hot
   * @param column the column being retrieved
   * @return the named column as a double
   */
  public double getDouble(int column) throws SQLException {
    String str;

    str = getString(column);
    if( str == null ) return 0;
    try {
      return Double.valueOf(str).doubleValue();
    }
    catch( NumberFormatException e ) {
      throw new SQLException(e.getMessage());
    }
  }  

  /**
   * JDBC draft specification method to retrieve a Numeric object from
   * the database.
   * @see java.sql.ResultSet#getNumeric
   * @exception SQLException things did not go so hot
   * @param column the column being retrieved
   * @param scale how many decimal digits after the floating point to maintain
   * @return the named column as a Numeric
   */
  public Numeric getNumeric(int column, int scale)
       throws SQLException {
    String str;

    str = getString(column);
    if( str == null ) return new Numeric(0, scale);
    else return new Numeric(str, scale);
  }

  /**
   * JDBC draft specification method to retrieve a decimal from
   * the database.
   * @see java.sql.ResultSet#getDecimal
   * @exception SQLException things did not go so hot
   * @param column the column being retrieved
   * @param scale how many decimal digits after the floating point to maintain
   * @return the named column as a Numeric
   */
  public Numeric getDecimal(int column, int scale)
      throws SQLException {
    String str;

    str = getString(column);
    if( str == null ) return new Numeric(0, scale);
    else return new Numeric(str, scale);
  }

  /**
   * JDBC draft specification method to return a byte array.
   * @see java.sql.ResultSet#getBytes
   * @exception SQLException thrown if something goes wrong
   * @param column the column being retrieved
   * @return a byte array that is the value of the column
   */

  public byte[] getBytes(int column) throws SQLException {
    String str;
    byte b[];

    str = getString(column);
    if( str == null ) return null;
    b = new byte[str.length() + 10];
    str.getBytes(0, str.length(), b, 0);
    return b;
  }

  /**
   * JDBC draft specification for retrieving a date column.
   * Can you say namespace pollution?  I knew you could.
   * @see java.sqlResultSet#getDate
   * @exception SQLException thrown in the event of problems
   * @param column the column being retrieved
   * @return the date value for the column
   */
  public java.sql.Date getDate(int column)
       throws SQLException {
    String str;

    str = getString(column);
    if( str == null ) return null;
    try {
      java.util.Date d;

      d = new java.util.Date(str);
      return new java.sql.Date(d.getYear(), d.getMonth(), d.getDate());
    }
    catch( Exception e ) {
      throw new SQLException("Data format error: " + e.getMessage());
    }
  }

  /**
   * JDBC draft specification method for retrieving a time from the database.
   * @see java.sql.ResultSet#getTime
   * @exception SQLException thrown in the event of troubles
   * @param column the column being retrieved
   * @return the column as a java.sql.Time object
   */
  public java.sql.Time getTime(int column)
       throws SQLException {
    String str;

    str = getString(column);
    if( str == null ) return null;
    try {
      java.util.Date d;

      d = new java.util.Date(str);
      return new java.sql.Time(d.getHours(), d.getMinutes(), d.getSeconds());
    }
    catch( Exception e ) {
      throw new SQLException("Data format error: " + e.getMessage());
    }
  }

  /**
   * JDBC draft specification method for retrieving a timestamp from
   * the database.
   * @see java.sql.ResultSet#getTimestamp
   * @exception SQLException thrown in the event of troubles
   * @param column the column being retrieved
   * @return the column as a java.sql.Timestamp object
   */
  public java.sql.Timestamp getTimestamp(int column)
       throws SQLException {
    String str;

    str = getString(column);
    if( str == null ) return null;
    try {
      java.util.Date d = new java.util.Date(str);
      
      return new java.sql.Timestamp(d.getYear(), d.getMonth(), d.getDate(),
				    d.getHours(), d.getMinutes(),
				    d.getSeconds(), 0);
    }
    catch( Exception e ) {
      throw new SQLException("Data format error: " + e.getMessage());
    }
  }

  /**
   * This is not currently supported.
   */
  public java.io.InputStream getAsciiStream(int column)
       throws SQLException {
    return null;
  }

  /**
   * This is not currently supported.
   */
  public java.io.InputStream getUnicodeStream(int column)
       throws SQLException {
    return null;
  }

  /**
   * This is not currently supported.
   */
  public java.io.InputStream getBinaryStream(int column)
       throws SQLException {
    return null;
  }
  
  /**
   * JDBC draft specification method for closing a result set.
   * This has no meaning to mSQL.
   * @see java.sql.ResultSet#close
   */
  public void close() throws SQLException {
  }

  /**
   * JDBC draft specification method for returning a cursor name.
   * mSQL does not support this feature.
   * @see java.sql.ResultSet#getCursorName
   * @return ""
   */
  public String getCursorName() throws SQLException {
    return "";
  }

  /**
   * JDBC draft specification method for returning meta-deta on a result
   * set.
   * @see java.sql.ResultSet#getMetaData
   * @exception SQLException thrown on error getting meta-data
   * @return ResultSetMetaData object containing result set info
   */
  public ResultSetMetaData getMetaData()
       throws SQLException {
    if( meta == null ) {
      meta = new iMsqlMetaResult(result);
    }
    return meta;
  }

  /**
   * JDBC draft specification method for retrieving data as objects.
   * @see java.sql.ResultSet#getObject
   * @exception SQLException in the event of retrieval! error
   * @param column the column desired
   * @param type the SQL data type of the field
   * @scale preceision for Numerics
   * @return the column specified as an Object
   */
  public Object getObject(int column, int type, int scale)
       throws SQLException {
    switch(type) {
    case Types.BIT:
      return new Boolean(getBoolean(column));

    case Types.TINYINT:
      return new Character((char)getInt(column));

    case Types.SMALLINT:
      return new Integer(getShort(column));

    case Types.INTEGER:
      return new Integer(getInt(column));

    case Types.BIGINT:
      return new Long(getLong(column));

    case Types.FLOAT:
      return new Float(getFloat(column));

    case Types.REAL:
      return new Float(getFloat(column));

    case Types.DOUBLE:
      return new Double(getDouble(column));

    case Types.NUMERIC:
      return getNumeric(column, scale);

    case Types.DECIMAL:
      return getNumeric(column, scale);

    case Types.CHAR:
      return getString(column);

    case Types.VARCHAR:
      return getString(column);

    case Types.LONGVARCHAR:
      return getString(column);

    case Types.DATE:
      return getDate(column);

    case Types.TIME:
      return getTime(column);

    case Types.TIMESTAMP:
      return getTimestamp(column);

    case Types.BINARY:
      return getBytes(column);

    case Types.VARBINARY:
      return getBytes(column);

    case Types.LONGVARBINARY:
      return getBytes(column);

    default:
      return null;
    }
  }

  /**
   * Same as above, except with a default scale to 0.
   */
  public Object getObject(int column, int type)
       throws SQLException {
    return getObject(column, type, 0);
  }

  /**
   * Same as above, except using the column's default SQL type.
   */
  public Object getObject(int column) throws SQLException {
    ResultSetMetaData meta = getMetaData();
    int type = meta.getColumnType(column);

    return getObject(column, type);
  }
    
  /**
   * JDBC 0.65 specification method for retrieving a column value
   * as a char based on the column name.
   * @see java.sql.ResultSet#getString
   * @param name the name of the column desired
   * @return the value of the column as a char
   */
  public String getString(String name) throws SQLException {
    return getString(findColumn(name));
  }

  /**
   * Returns the column as a byte based on column name
   */
  public byte getByte(String columnName) throws SQLException {
    return getByte(findColumn(columnName));
  }
  
  /**
   * Get the value of a BIT column in the current row
   *
   * @param columnName is the SQL name of the column
   * @return the column value; if isNull the value is false
   */
  public boolean getBoolean(String columnName) throws SQLException {
    return getBoolean(findColumn(columnName));
  }
  
  /**
   * Get the value of a short by column name
   *
   * @param columnName is the SQL name of the column
   * @return the column value; if isNull the value is 0
   */
  public short getShort(String columnName) throws SQLException {
    return getShort(findColumn(columnName));
  }

  /**
   * Get the value of a INTEGER column in the current row
   *
   * @param columnName is the SQL name of the column
   * @return the column value; if isNull the value is 0
   */
  public int getInt(String columnName) throws SQLException {
    return getInt(findColumn(columnName));
  }
  
  /**
   * Get the value of a long column in the current row
   *
   * @param columnName is the SQL name of the column
   * @return the column value; if isNull the value is 0
   */
  public long getLong(String columnName) throws SQLException {
    return getLong(findColumn(columnName));
  }

  /**
   * Get the value of a FLOAT column in the current row
   *
   * @param columnName is the SQL name of the column
   * @return the column value; if isNull the value is 0
   */
  public float getFloat(String columnName) throws SQLException {
    return getFloat(findColumn(columnName));
  }

  /**
   * Get the value of a DOUBLE column in the current row
   *
   * @param columnName is the SQL name of the column
   * @return the column value; if isNull the value is 0
   */
  public double getDouble(String columnName) throws SQLException {
    return getDouble(findColumn(columnName));
  }
  
  /**
   * Get the value of a NUMERIC column in the current row
   *
   * @param columnName is the SQL name of the column
   * @return the column value; if isNull the value is null
   */
  public Numeric getNumeric(String columnName, int scale) throws SQLException {
    return getNumeric(findColumn(columnName), scale);
  }

  /**
   * Get the value of a BINARY column in the current row
   *
   * @param columnName is the SQL name of the column
   * @return the column value; if isNull the value is null
   */
  public byte[] getBytes(String columnName) throws SQLException {
    return getBytes(findColumn(columnName));
  }
  
  /**
   * Get the value of a DATE column in the current row
   *
   * @param columnName is the SQL name of the column
   * @return the column value; if isNull the value is null
   */
  public java.sql.Date getDate(String columnName) throws SQLException {
    return getDate(findColumn(columnName));
  }
  
  /**
   * Get the value of a TIME column in the current row
   *
   * @param columnName is the SQL name of the column
   * @return the column value; if isNull the value is null
   */
  public java.sql.Time getTime(String columnName) throws SQLException {
    return getTime(findColumn(columnName));
  }
  
  /**
   * Get the value of a TIMESTAMP column in the current row
   *
   * @param columnName is the SQL name of the column
   * @return the column value; if isNull the value is null
   */
  public java.sql.Timestamp getTimestamp(String columnName)
       throws SQLException {
    return getTimestamp(findColumn(columnName));
  }
  
  /**
   * Very large ascii values in a LONGVARCHAR column can
   * be read in small chunks from a java.io.InputStream.
   *
   * <P><B>Note:</B> All the data in the returned stream must
   * be read prior to getting the value of any other column. The
   * next call to a get method implicitly closes the stream.
   *
   * @param columnName is the SQL name of the column
   * @return the java input stream which contains the ascii value 
   */
  public java.io.InputStream getAsciiStream(String columnName)
       throws SQLException {
    return getAsciiStream(findColumn(columnName));
  }

  /**
   * Very large unicode values in a LONGVARCHAR column can
   * be read in small chunks from a java.io.InputStream.
   *
   * <P><B>Note:</B> All the data in the returned stream must
   * be read prior to getting the value of any other column. The
   * next call to a get method implicitly closes the stream.
   *
   * @param columnName is the SQL name of the column
   * @return the java input stream which contains the unicode value 
   */
  public java.io.InputStream getUnicodeStream(String columnName)
       throws SQLException {
    return getUnicodeStream(findColumn(columnName));
  }

  /**
   * Very large binary values in a LONGVARBINARY column can
   * be read in small chunks from a java.io.InputStream.
   *
   * <P><B>Note:</B> All the data in the returned stream must
   * be read prior to getting the value of any other column. The
   * next call to a get method implicitly closes the stream.
   *
   * @param columnName is the SQL name of the column
   * @return the java input stream which contains the binary value */
  public java.io.InputStream getBinaryStream(String columnName)
       throws SQLException {
    return getBinaryStream(findColumn(columnName));
  }

  /**
   * Get the value of a column as an object; an integral value is
   * returned as its java.lang equivalent object; a LONGVARCHAR or
   * LONGVARBINARY value is returned as a java.io.InputStream.
   *
   * @param columnName the SQL column name
   * @param sqlType SQL type code defined by java.sql.Types
   * @return the parameter value; if isNull the value is null 
   */
  public Object getObject(String columnName, int sqlType, int scale)
       throws SQLException {
    return getObject(findColumn(columnName), sqlType, scale);
  }

  /**
   * Same as above, except defaulting scale to 0.
   */
  public Object getObject(String columnName, int type)
       throws SQLException {
    return getObject(findColumn(columnName), type, 0);
  }

  /**
   * Same as above, except returning the default SQL type
   */
  public Object getObject(String columnName) throws SQLException {
    return getObject(findColumn(columnName));
  }
  
  /**
   * column name counter part for other isNull()
   */
  public boolean isNull(String name) throws SQLException {
    return isNull(findColumn(name));
  }

  /**
   * Given a column name, this method returns the column number for that
   * name.  Column name to number mappings are kept inside a Hashtable.
   * Applications that do not need the overhead of this calculation are
   * not penalized since the mapping only occurs on the first attempt to
   * access a column number by name.
   * @exception java.sql.SQLException thrown if a bad name is passed
   * @param name the name of the column desired
   * @return the column number, 1 being the first column
   */
  public int findColumn(String name) throws SQLException {
    Integer num;
    
    if( column_map == null ) {
      ResultSetMetaData m;
      int i, maxi;
      
      m = getMetaData();
      column_map = new Hashtable(maxi = m.getColumnCount());
      for(i=0; i<maxi; i++) {
	column_map.put(m.getColumnName(i), new Integer(i+1));
      }
    }
    num = (Integer)column_map.get(name);
    if( num == null ) {
      throw new SQLException("Invalid column name: " + name);
    }
    return num.intValue();
  }
  /**
   * JDBC draft specification for getting the chain of warnings for this
   * statement.
   * @see java.sql.Statement#getWarnings
   * @return the chain of warnings
   */
  public SQLWarning getWarnings() throws SQLException {
    return null;
  }

  /**
   * JDBC draft specification for clearing the warning chain.
   * @see java.sql.Statement#clearWarnings
   */
  public void clearWarnings() throws SQLException {
  }
}
