(* Copyright (C) 1992, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)

(* File: ReturnStmt.m3                                         *)
(* Last modified on Mon Mar  2 11:11:37 PST 1992 by kalsow     *)
(*      modified on Thu Dec  5 17:22:32 PST 1991 by muller     *)

MODULE ReturnStmt;

IMPORT Expr, Error, Type, AssignStmt, Token, Scanner;
IMPORT Variable, Void, Temp, Marker, Stmt, StmtRep;

TYPE
  P = Stmt.T OBJECT
        expr    : Expr.T;
      OVERRIDES
        check    := Check;
	compile  := Compile;
        outcomes := GetOutcome;
      END;

PROCEDURE Parse (READONLY fail: Token.Set): Stmt.T =
  VAR p := NEW (P);
  BEGIN
    StmtRep.Init (p);
    p.expr := NIL;
    Scanner.Match (Token.T.tRETURN, fail, Token.ExprStart);
    IF (Scanner.cur.token IN Token.ExprStart) THEN
      p.expr := Expr.Parse (fail);
    END;
    RETURN p;
  END Parse;

PROCEDURE Check (p: P;  VAR cs: Stmt.CheckState) =
  VAR isVoid: BOOLEAN;  t: Type.T;  v: Variable.T;
  BEGIN
    Expr.TypeCheck (p.expr, cs);
    IF NOT Marker.ReturnOK () THEN
      Error.Msg ("RETURN not in a procedure");
      RETURN ;
    END;
    Marker.ReturnVar (t, v);
    isVoid := Type.IsEqual (t, Void.T, NIL);
    IF (p.expr = NIL) THEN
      IF (NOT isVoid) THEN Error.Msg ("missing return result") END;
    ELSIF (isVoid) THEN
      Error.Msg ("procedure does not have a return result");
    ELSE
      p.expr := AssignStmt.CheckRHS (t, p.expr, cs);
    END;
  END Check;

PROCEDURE Compile (p: P): Stmt.Outcomes =
  VAR x: Temp.T;
  BEGIN
    IF (p.expr # NIL) THEN
      x := Expr.Compile (p.expr);
      Marker.EmitReturn (x, Expr.TypeOf (p.expr));
      Temp.Free (x);
    ELSE
      Marker.EmitReturn (NIL, NIL);
    END;
    RETURN Stmt.Outcomes {Stmt.Outcome.Returns};
  END Compile;

PROCEDURE GetOutcome (<*UNUSED*> p: P): Stmt.Outcomes =
  BEGIN
    RETURN Stmt.Outcomes {Stmt.Outcome.Returns};
  END GetOutcome;

BEGIN
END ReturnStmt.
