(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)

(* Created by stolfi on Mon Nov  7 21:46:31 1988               *)
(* Last modified on Tue Feb 11 21:39:50 PST 1992 by muller     *)
(*      modified on Thu Oct 25  8:57:27 PDT 1990 by stolfi     *)

MODULE CIE;

IMPORT RGB;
FROM RGBCIE IMPORT
  rX, rY, rZ, gX, gY, gZ, bX, bY, bZ,
  Xr, Xg, Xb, Yr, Yg, Yb, Zr, Zg, Zb;

PROCEDURE XYZFromYxy (READONLY yxy: Yxy): T =
  VAR xyz: XYZ;
  BEGIN
    IF yxy = Undefined THEN RETURN Undefined END;
    xyz[1] := yxy[0];
    IF yxy[0] = 0.0 THEN
      xyz[0] := 0.0;
      xyz[2] := 0.0;
    ELSE
      (* Should guard against spurious overflow *)
      xyz[0] := yxy[1] * yxy[0] / yxy[2];
      xyz[2] := (1.0 - yxy[1] - yxy[2]) * yxy[0] / yxy[2];
    END;
    RETURN xyz
  END XYZFromYxy;

PROCEDURE YxyFromXYZ (READONLY xyz: XYZ): Yxy =
  VAR yxy: Yxy; w: REAL;
  BEGIN
    IF xyz = Undefined THEN RETURN Undefined END;
    yxy[0] := xyz[1];
    w := xyz[0] + xyz[1] + xyz[2];
    IF w = 0.0 THEN
      yxy[1] := 0.0;
      yxy[2] := 0.0;
    ELSE
      (* Should guard against spurious overflow *)
      yxy[1] := xyz[0] / w;
      yxy[2] := xyz[1] / w;
    END;
    RETURN yxy
  END YxyFromXYZ;

PROCEDURE XYZFromYuv (READONLY yuv: Yuv): T =
  VAR xyz: XYZ;
  BEGIN
    IF yuv = Undefined THEN RETURN Undefined END;
    xyz[1] := yuv[0];
    IF yuv[0] = 0.0 THEN
      xyz[0] := 0.0;
      xyz[2] := 0.0;
    ELSE
      (* Should guard against spurious overflow *)
      xyz[0] := 2.25 * yuv[1] * yuv[0] / yuv[2];
      xyz[2] := (3.0 - 5.0 * yuv[2] - 0.75 * yuv[1])
        * yuv[0] / yuv[2];
    END;
    RETURN xyz
  END XYZFromYuv;

PROCEDURE YuvFromXYZ (READONLY xyz: XYZ): Yuv =
  VAR yuv: Yuv; w: REAL;
  BEGIN
    IF xyz = Undefined THEN RETURN Undefined END;
    yuv[0] := xyz[1];
    w := xyz[0] + 15.0 * xyz[1] + 3.0 * xyz[2];
    IF w = 0.0 THEN
      yuv[1] := 0.0;
      yuv[2] := 0.0
    ELSE
      (* Should guard against spurious overflow *)
      yuv[1] := 4.0 * xyz[0] / w;
      yuv[2] := 9.0 * xyz[1] / w;
    END;
    RETURN yuv
  END YuvFromXYZ;

(**********************************************************)
(*                                                        *)
(* CONNECTION TO RGB COORDINATES                          *)
(*                                                        *)
(**********************************************************)

PROCEDURE RGBFromXYZ (READONLY xyz: XYZ): RGB.T =
  BEGIN
    IF xyz = Undefined THEN RETURN RGB.Undefined END;
    RETURN RGB.T{
      Xr * xyz[0] + Yr * xyz[1] + Zr * xyz[2],
      Xg * xyz[0] + Yg * xyz[1] + Zg * xyz[2],
      Xb * xyz[0] + Yb * xyz[1] + Zb * xyz[2]
    }
  END RGBFromXYZ;

PROCEDURE XYZFromRGB (READONLY rgb: RGB.T): XYZ =
  BEGIN
    IF rgb = RGB.Undefined THEN RETURN Undefined END;
    RETURN XYZ{
      rX * rgb[0] + gX * rgb[1] + bX * rgb[2],
      rY * rgb[0] + gY * rgb[1] + bY * rgb[2],
      rZ * rgb[0] + gZ * rgb[1] + bZ * rgb[2]
    }
  END XYZFromRGB;
  
BEGIN

END CIE.

