(* Copyright (C) 1992, Digital Equipment Corporation *)
(* All rights reserved. *)
(* See the file COPYRIGHT for a full description. *)
(* Created by stolfi on Thu Sep 15 02:38:28 1988 *)
(* Last modified on Thu Jun 11 19:10:21 1992 by mhb    *)
(*      modified on Tue Jun 2 13:44:18 PDT 1992 by stolfi *)
(* modified on Tue Feb 11 21:39:46 PST 1992 by muller *)


(* The "Intensity" interface defines data types and tools for
   dealing with light intensity tables. *)

INTERFACE Intensity;

TYPE T = REAL;

PROCEDURE Equal (a, b : T;
                 err  : REAL := 1.0e-06;
                 small: REAL := 1.0e-30  ): BOOLEAN;
(* Returns "TRUE" if "a" and "b" are approximately equal within
   the relative error "err" or the absolute error "err*small".
   More precisely, if
|       ABS(a - b)/MAX(ABS(a), ABS(b), small) <= eps
   *)

(* In addition to the ordinary intensities, some applications
   need a special value meaning ``undefined'', ``don't care'',
   and so on.

   The "Undefined" value below is meant for this purpose.  It
   lies well outside the unit interval, and is unlikely to be
   generated by coincidence. *)

CONST Undefined: T = 8880888.0;

(* Unless said otherwise, passing "Undefined" to a procedure that
   expects an intensity is an unchecked runtime error; in
   general, the result will be neither "Undefined" nor a valid
   intensity. *)

CONST Default: T = 0.0;
(* A reasonable default value, for the sake of generic modules *)

TYPE
  Byte = BITS 8 FOR [0 .. 255];
  Bytes = ARRAY [0 .. BYTESIZE(T) - 1] OF Byte;

PROCEDURE ToBytes (x: T): Bytes;
(* A safe "LOOPHOLE" of "x" into "ARRAY OF BYTE". *)

END Intensity.

