(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)

(* Created by stolfi on Wed Oct 24  2:20:57 PDT 1990           *)
(* Last modified on Tue Feb 11 21:39:41 PST 1992 by muller     *)
(*      modified on Thu Oct 25  6:53:10 PDT 1990 by stolfi     *)

INTERFACE RGBDist;

(* Color distances and color matching.
  
  This interface exports procedures for computing distances between
  color values and finding the nearest matches to a given color among
  a set of RGB values.
  
  Index: colors; graphics, colors; display, color.
*)

IMPORT RGB;

(*
  The perceptual difference between two colors is a very complex
  quantity.  It depends not only on the two colors, but also on the
  size of the areas covered by them, the distance (in time and space)
  between those two areas, on the color of the surrounding background,
  and so on.  Even for a fixed background and geometry, the distance
  is varies with the RGB coordinates in a highly non-linear fashion.
  
  As a modest step towards addressing these difficulties, the distance
  functions below take a /lumWeight/ parameter that specifies the
  importance of brightness differences relative to hue and saturation
  differences.  Generally speaking, brightness differences are more
  significant when the two colored areas are adjacent to each other
  (or the two colors flash in quick succession over the same area);
  the default /lumWeight/ is adjusted for this case.  brightness becomes
  less important, relative to hue and saturation, as the separation
  in time or space between the two areas increases; for these
  situations, one should use a smaller /lumWeight/.
  
  The visual difference between two colored patches decreases also
  when the patches get smaller, or when they are surrounded by a
  background of a contrasting color.  The procedures below do not thake
  these effects into account.  *)

PROCEDURE Quick(READONLY a, b: RGB.T; lumWeight: REAL := 1.0): REAL;
  (*
    A fast but rather inaccurate color distance function. *)

PROCEDURE Perceptual(READONLY a, b: RGB.T; lumWeight: REAL := 1.0): REAL;
  (* 
    A moderately accurate color distance function, defined as
|      LDW.Dist(LDW.LDWFromRGB(a), LDW.LDWFromRGB(b). lumWeight)
    *)    
    
END RGBDist.
