(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)

(* Created by stolfi on Sun Aug  6 14:31:30 1989               *)
(* Last modified on Tue Feb 11 21:39:41 PST 1992 by muller     *)
(*      modified on Wed Nov 20 18:58:23 PST 1991 by stolfi     *)

INTERFACE RGBIO;

(* Reading and printing RGB triples in symbolic list (Sx) format.

  This interface provides routines for printing and parsing color
  values (real-valued RGB triples) and color tables, in a format
  compatible with that of sybolic expressions (Sx.def).
  
  Index: colors; graphics, colors; display, color;
         colors, colormaps; images, colormaps; 
         symbolic expressions; expressions, symbolic;
         colors, printing and parsing;
         printing, colors; parsing, colors; colors, printing and parsing.

*)

IMPORT Rd, Wr, Thread, Scan, RGB;

TYPE NAT = CARDINAL;

(* 
  This interface defines a representation for color values
  as ASCII strings.

  A color value (an RGB.T) is represented as

    - a list of three RGB coordinates, normally in [0.0 _ 1.0]; or

    - a color name (a single identifier or a quoted text string),
         following the conventions of ColorName.def; or

    - an empty list (), signifying RGB.Undefined.

  Examples:

    (0 0 0)
    (0.250 0.333 0.180)
    ()
    Red
    DarkYellow
    "vey very dim somewhat dull slightly pale purplish turquoise"
    Undefined
    NIL

  A color table (an RGB.ArrT) is represented as a list of the form

|      (
|        ( index1  color1 )
|        ( index2  color2 )
|        ...
|        ( indexn  colorn )
|      )

  where each /indexi/ is an index into the array, and /colori/ is a
  description of the color stored there, as described above.

  On input, extra white space and newlines are allowed everywhere,
  except inside numbers and unquoted color names.  Numbers must be
  compatible with the %f format of SCANF.  Array entries can be
  given in any order; missing entries are assumed to be RGB.Undefined.

  On output, RGB values are always printed as triples (r.rrrr g.gggg
  b.bbbb), or the empty list (), with no leading or trailing blanks
  or newlines.  The coordinates are printed with four decimals after
  the period.  Arrays are printed one entry per line, in order,
  omitting all RGB.Undefined entries.
*)


PROCEDURE Write (wr: Wr.T; READONLY rgb: RGB.T) 
  RAISES {Wr.Failure, Thread.Alerted};
  (*
    Writes an RGB triple to /wr/. *)

PROCEDURE ToText (READONLY rgb: RGB.T): TEXT RAISES {};
  (*
    Converts /rgb/ to a TEXT *)

PROCEDURE WriteArray (wr: Wr.T; READONLY table: ARRAY OF RGB.T)
  RAISES {Wr.Failure, Thread.Alerted};
  (*
    Writes a table of RGB values to /wr/. *)

PROCEDURE Read (rd: Rd.T): RGB.T
  RAISES {Scan.BadFormat, Rd.EndOfFile, Rd.Failure, Thread.Alerted};
  (*
    Reads an RGB.T triple from /rd/, in the format described above.
    The procedure raises Rd.EndOfFile if the reader ends before
    the first non-blank character is found.  Raises Scan.BadFormat if the
    input is malformed or the reader is exhausted halfway through the
    parsing.  *)

PROCEDURE FromText (text: TEXT): RGB.T 
  RAISES {Scan.BadFormat};
  (*
    Parses /text/ into an RGB value. *)

PROCEDURE ReadArray (rd: Rd.T; maxSize: NAT := LAST (NAT)): REF ARRAY OF RGB.T
  RAISES {Scan.BadFormat, Rd.EndOfFile, Rd.Failure, Thread.Alerted};
  (*
    Reads a table of RGB values from /rd/, in the format described above.
    The procedure raises Rd.EndOfFile if the reader is exhausted before
    the opening '('.  Raises Scan.BadFormat if the input is malformed,
    if the reader is exhausted halfway through the parsing, if some
    entry has an index >= maxSize, or if two entries have the same
    index.  *)

END RGBIO.

