(* Copyright (C) 1990, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)

(* Last modified on Thu Jan 30 14:47:00 PST 1992 by kalsow     *)
(*      modified on Wed Jul  3 04:15:39 1991 by muller         *)

INTERFACE RTHeap;

(* This interface provides safe access to the storage allocator
   and garbage collector. *)

FROM RT0 IMPORT Typecode;

(*-----------------------  allocation  ------------------------*)

PROCEDURE Allocate (tc: Typecode): REFANY;
(* allocate in the traced heap a properly initialized referent for a
   reference with typecode tc.  If TYPECODE (T) = tc, then Allocate (tc)
   has the same effect as NEW (T).  It is a checked runtime error if tc
   is not a proper typecode, tc corresponds to an untraced type, or the 
   referent is an open array. *)

PROCEDURE AllocateUntraced (tc: Typecode): UNTRACED ROOT;
(* allocate in the untraced heap a properly initialized referent for a
   reference with typecode tc.  If TYPECODE (T) = tc, then
   AllocateUntraced (tc) has the same effect as NEW (T).  It is a checked
   runtime error if tc is not a proper typecode, tc corresponds to a traced
   type, or the referent is an open array. *)

PROCEDURE AllocateOpenArray (tc: Typecode; 
                             READONLY s: ARRAY OF INTEGER): REFANY;
(* allocate in the traced heap a properly initialized open array referent
   for a reference with typecode tc.  If TYPECODE (T) = tc, then
   AllocateOpenArray(tc, s) has the same effect as
     NEW (T, s[0], s[1], ..., s[HIGH(s)]).
   It is a checked runtime error if tc is not a proper typecode, tc
   corresponds to an untraced type, or the referent is not an open array. *)


PROCEDURE AllocateUntracedOpenArray (tc: Typecode;
                                     READONLY s: ARRAY OF INTEGER): ADDRESS;
(* allocate in the untraced heap a non-initialized open array referent
   for a reference with typecode tc.  If TYPECODE (T) = tc, then
   AllocateOpenArray(tc, s) has the same effect as
     NEW (T, s[0], s[1], ..., s[HIGH(s)]).
   It is a checked runtime error if tc is not a proper typecode, tc
   corresponds to a traced type, or the referent is not an open array. *)


PROCEDURE AllocateUntracedRef (tc: Typecode): ADDRESS;

PROCEDURE Duplicate (r: REFANY): REFANY;
(* return a reference to a new allocated copy of the referent, not
   recursively applied to internal references.  Duplicate(r) is
   equivalent to Allocate (TYPECODE (r)).  *)


(* These two routines do not check as much as the standard versions and 
   do not initialize the allocated memory to 0 *)

PROCEDURE FastAllocate (tc: Typecode): REFANY;

PROCEDURE FastAllocateOpenArray (tc: Typecode; 
                                 READONLY s: ARRAY OF INTEGER): REFANY;

(*-----------------------  collector  --------------------------*)

(* The collector may run at any time that it is enabled.  Each call
   to DisableCollection increments a counter, each call to
   EnableCollection decrements that counter.  When the
   counter is zero, the collector is enabled.  Initially
   the counter is zero. *)

PROCEDURE DisableCollection ();
  (* disables the collector (and hence REFs being moved) *)

PROCEDURE EnableCollection ();
  (* enables the collector *)


(* Each traced referent has a freeze counter associated with it.  The
   traced referent can be moved by the garbage collector iff this
   counter is 0.  When a referent is created, its counter is 0.  Of
   course, we do not store the counter for every referent; only for those
   with a non-null counter.  *)

PROCEDURE FreezeRef (r: REFANY) RAISES {};
  (* Increment the freeze counter for the referent pointed by r.  It
     is a checked runtime error if r is NIL. *)

PROCEDURE UnfreezeRef (r: REFANY) RAISES {};
  (* Decrement the freeze counter for the referent pointer by r.  It
     is a checked runtime error if r is NIL or its counter is 0 before the
     call. *)

(*----------------------  heap data layout  -------------------------*)

PROCEDURE GetDataAdr (r: REFANY): ADDRESS;
(* return the address of the data part of the referent *)

PROCEDURE GetDataSize (r: REFANY): CARDINAL;
(* return the size in bytes of the data part of the referent,
  the method suite pointer or open array shape is not included *)

(*----------------------  open arrays -------------------------*)

TYPE ArrayShape = UNTRACED REF ARRAY [0..999] OF INTEGER;

PROCEDURE GetShape (r: REFANY;  VAR nDimensions: INTEGER;  VAR s: ArrayShape);
(* if r is a reference to an open array, the number of open dimensions,
   nDimensions, and size of each dimension, s, is returned.  The array's
   shape vector is valid as long as r exists.  If r is not a reference
   to an open array, nDimensions = 0 and s is undefined.
  It is an unchecked runtime
   error to modify s^, to free s, or to use it after r has been garbage
   collected. *)

PROCEDURE GetNDimensions (tc: Typecode): CARDINAL;
(* if tc corresponds to a reference to an open array type, the
   number of open dimensions of that type is returned.  Otherwise,
   0 is returned. *)

END RTHeap.
